// xgui 0.0.4 / 2002-08-07
//	picker.cpp
//
//	http://606u.dir.bg/
//	606u@dir.bg

#include "_p.h"

#include "picker.h"
#include "palette.h"

#include "resource.h"


namespace xgui {

// picker's static members initialization
const char	*picker::class_name = "xgui_picker";
WNDCLASS	picker::window_class = { 0 };
ATOM		picker::class_atom = 0;

HCURSOR		picker::picker_cursor = NULL;


picker::picker (void)
	: base ()
{
	visual_mode = -1;
	additional_component = -1;
	x_pos = -1;
	y_pos = -1;
	max_x = -1;
	max_y = -1;
	old_x = -1;
	old_y = -1;
	should_repaint = true;
	bmp_handle = NULL;
}


picker::~picker ()
{
	if (bmp_handle)
		DeleteObject (bmp_handle), bmp_handle = NULL;

	// controller owns the palette
}


bool
picker::register_class (
	IN HINSTANCE library_instance)
{
	picker_cursor = LoadCursor (library_instance, MAKEINTRESOURCE (102));	// 102 = IDC_PICKER

	setup_wndclass (window_class, library_instance, class_name);
	if (picker_cursor)
		window_class.hCursor = NULL;
	else
	{
		// error; set arrow as default cursor
		debug_message ("picker::register_class: cursor load failed; default is used.");
		picker_cursor = LoadCursor (NULL, IDC_CROSS);
	}

	return (wndclass::reg (class_name, &window_class, &class_atom));
}


bool
picker::unregister_class (
	IN HINSTANCE library_instance)
{
	if (picker_cursor)
		DestroyCursor (picker_cursor);

	return (wndclass::unreg (library_instance, class_name));
}


bool
picker::setup (
	IN HWND hwnd,
	IN LPCREATESTRUCT pcs)
{
	if (base::setup (hwnd, pcs))
	{
		// default to rgb red picker
		visual (modes::rgb_red);

		// setup window
		return (size_changed (pcs->cx, pcs->cy));
	}

	return (false);
}


void
picker::mouse_move (
	IN short x,
	IN short y,
	IN int /* modifiers */)
{
	if (tracking)
	{
		SetCursor (picker_cursor);

		// make sure x and y are inside blend_rect
		in_range (x, blend_rect.left, blend_rect.right);
		in_range (y, blend_rect.top, blend_rect.bottom);

		// scale x and y in [0, max_x] and [0, max_y];
		//	use ceil, otherwise there is a difference between cursor position
		//	and selection circle drawn
		x = (short) ceil ((x - blend_rect.left) * (double) max_x / (blend_rect.right - blend_rect.left));
		y = (short) ceil ((blend_rect.bottom - y) * (double) max_y / (blend_rect.bottom - blend_rect.top));
		if (x == x_pos && y == y_pos)
			// don't redraw if not necessary
			return;

		x_pos = x;
		y_pos = y;
		InvalidateRect (hwnd, NULL, FALSE);

		debug_state ("mouse_move, x = %d in [0, %d], y = %d in [0, %d]",
			x_pos, max_x, y_pos, max_y);

		// color has changed
		notify_parent (messages::drag | color ());
	}
	else
	{
		POINT	pt = { x, y };
		if (PtInRect (&interaction_area, pt))
			SetCursor (picker_cursor);
		else
			SetCursor (LoadCursor (NULL, IDC_ARROW));
	}
}


void
picker::mouse_up (
	IN enum mouse_buttons button,
	IN short x,
	IN short y,
	IN int modifiers)
{
	if (button == button_left && tracking)
		// notify parent window
		notify_parent (messages::release | color ());
	base::mouse_up (button, x, y, modifiers);
}


void
picker::mouse_wheel (
	IN short /* delta */)
{
}


void
picker::key_down (
	IN int key_code,
	IN int /* modifiers */)
{
	int	x, y;

	// don't process if mouse is being tracked
	if (tracking)
		return;

	x = x_pos;
	y = y_pos;
	switch (key_code)
	{
	case	VK_UP:
		y++; break;
	case	VK_LEFT:
		x--; break;
	case	VK_DOWN:
		y--; break;
	case	VK_RIGHT:
		x++; break;
	case	VK_PRIOR:
		x = max_x;
		y = max_y;
		break;
	case	VK_NEXT:
		x = max_x;
		y = 0;
		break;
	case	VK_HOME:
		x = 0;
		y = max_y;
		break;
	case	VK_END:
		x = 0;
		y = 0;
		break;
	default:
		return;
	}

	in_range (x, 0, max_x);
	in_range (y, 0, max_y);
	if (x == x_pos && y == y_pos)
		// don't redraw if not necessary
		return;

	x_pos = x;
	y_pos = y;
	InvalidateRect (hwnd, NULL, FALSE);

	debug_state ("key_down, x=%d in [0, %d], y=%d in [0, %d]",
		x_pos, max_x, y_pos, max_y);

	// color has changed
	notify_parent (messages::release | color ());
}


void
picker::paint (
	IN LPPAINTSTRUCT pnt)
{
	HDC		hdc;

	// create compatible DC
	hdc = CreateCompatibleDC (pnt->hdc);
	if (hdc != NULL)
	{
		HBRUSH	bg_fill;

		SelectObject (hdc, bmp_handle);

		if (should_repaint || CONTRAST_MARKER)
		{
			/* since this is the first time picker will be drawn,
			   old point should not be restored */
			old_x = -9999;
			old_y = -9999;

			// paint control's background
			bg_fill = CreateSolidBrush (GetSysColor (COLOR_BTNFACE));
			SelectObject (hdc, GetStockObject (NULL_PEN));
			SelectObject (hdc, bg_fill);
			Rectangle (hdc, 0, 0, frame.right + 1, frame.bottom + 1);
			SelectObject (hdc, GetStockObject (WHITE_BRUSH));
			DeleteObject (bg_fill);

			// paint blend and frame
			panel::draw (hdc, &panel_rect,
				panel::sunken | panel::blackbox);
			GdiFlush ();
			blend ();
		}
		else
		{
			// clear (possibly) previous focus rectangle
			HPEN	pen_frame = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNFACE));

			SelectObject (hdc, pen_frame);
			SelectObject (hdc, GetStockObject (NULL_BRUSH));
			Rectangle (hdc, frame.left, frame.top,
				frame.right, frame.bottom);
			SelectObject (hdc, GetStockObject (BLACK_PEN));
			DeleteObject (pen_frame);
		}

		// paint a focus rectangle if necessary
		if (GetFocus () == hwnd)
			DrawFocusRect (hdc, &frame);

		// current position
		if (CONTRAST_MARKER)
		{
			int	x, y, size;
			COLORREF	c;
			int	val;
			HPEN	marker, old;

			// arc size
			size = MARKER_SIZE;

			IntersectClipRect (hdc, blend_rect.left, blend_rect.top,
				blend_rect.right + 1, blend_rect.bottom + 1);

			// current position
			x = x_pos * (blend_rect.right - blend_rect.left) / max_x + blend_rect.left;
			y = blend_rect.bottom - y_pos * (blend_rect.bottom - blend_rect.top) / max_y;

			c = color ();
			val = max (max (GetRValue (c), GetGValue (c)), GetBValue (c));
			if (val < 128)
				marker = CreatePen (PS_SOLID, 1, RGB (255, 255, 255));
			else
				marker = CreatePen (PS_SOLID, 1, RGB (0, 0, 0));
			old = (HPEN) SelectObject (hdc, marker);
			Arc (hdc, x - size, y - size, x + size + 1, y + size + 1,
				x, y - size, x, y - size);
//			Rectangle (hdc, x - 1, y - 1, x + 2, y + 2);
			SelectObject (hdc, old);
			DeleteObject (marker);
		}
		else
		{
			int	x, y, size;

			// arc size
			size = MARKER_SIZE;

			SelectObject (hdc, GetStockObject (WHITE_PEN));
			IntersectClipRect (hdc, blend_rect.left, blend_rect.top,
				blend_rect.right + 1, blend_rect.bottom + 1);
			SetROP2 (hdc, R2_XORPEN);

			// current position
			x = x_pos * (blend_rect.right - blend_rect.left) / max_x + blend_rect.left;
			y = blend_rect.bottom - y_pos * (blend_rect.bottom - blend_rect.top) / max_y;
			Arc (hdc, x - size, y - size, x + size + 1, y + size + 1,
				x, y - size, x, y - size);

			// old position
			x = old_x * (blend_rect.right - blend_rect.left) / max_x + blend_rect.left;
			y = blend_rect.bottom - old_y * (blend_rect.bottom - blend_rect.top) / max_y;
			Arc (hdc, x - size, y - size, x + size + 1, y + size + 1,
				x, y - size, x, y - size);

			old_x = x_pos;
			old_y = y_pos;

			// restore raster ops mode
			SetROP2 (hdc, R2_COPYPEN);
		}

		// bitblt and delete cache DC
		GdiFlush ();
		BitBlt (pnt->hdc, 0, 0, frame.right, frame.bottom, hdc, 0, 0, SRCCOPY);
		DeleteDC (hdc);

		// blend is cached in the bitmap until next mode change or component change
		should_repaint = false;
	}
}


bool
picker::size_changed (
	IN int hsize,
	IN int vsize)
{
	if (!base::size_changed (hsize, vsize))
		return (false);

	SetRect (&panel_rect, 2, 2, hsize - 3, vsize - 3);
	InflateRect (&panel_rect,
		-((style & 0x0F00) >> 8) - 1,
		-((style & 0x0F00) >> 8) - 1);
	
	CopyRect (&blend_rect, &panel_rect);
	InflateRect (&blend_rect, -2, -2);
	CopyRect (&interaction_area, &blend_rect);

	// delete cache bitmap if window is resized
	if (bmp_handle != NULL)
		DeleteObject (bmp_handle);

	// create cache bitmap
	bmp_info.bmiHeader.biSize = sizeof (BITMAPINFO);
	bmp_info.bmiHeader.biWidth = hsize;
	bmp_info.bmiHeader.biHeight = vsize;
	bmp_info.bmiHeader.biPlanes = 1;
	bmp_info.bmiHeader.biBitCount = 32;
	bmp_info.bmiHeader.biCompression = BI_RGB;
	bmp_info.bmiHeader.biSizeImage = hsize * vsize * 32 / 8;
	bmp_info.bmiHeader.biXPelsPerMeter =
		bmp_info.bmiHeader.biYPelsPerMeter = 72 * 2 * 1000;
	bmp_info.bmiHeader.biClrUsed = 0;
	bmp_info.bmiHeader.biClrImportant = 0;
	bmp_data = NULL;
	bmp_handle = CreateDIBSection (
		NULL, &bmp_info, DIB_RGB_COLORS, (void **) &bmp_data, NULL, 0);
	if (bmp_handle == NULL)
	{
		debug_message ("picker::size_changed: CreateDIBSection failed.");
		return (false);
	}

	// blend should be drawn, because control size has changed
	should_repaint = true;

	return (true);
}


LRESULT
picker::control_message (
	IN enum control_codes message,
	IN WPARAM wparam,
	IN LPARAM lparam)
{
	LRESULT		result;

	// ask base class first
	result = base::control_message (message, wparam, lparam);
	if (result)
		return (result);

	result = 0;
	switch (message)
	{
	case	messages::visual_get:		return (visual ());
	case	messages::visual_set:		visual (lparam); break;
	case	messages::component_get:	return (component ());
	case	messages::component_set:	component (lparam); break;
	case	messages::xpos_get:			return (xpos ());
	case	messages::xpos_set:			xpos (lparam); break;
	case	messages::ypos_get:			return (ypos ());
	case	messages::ypos_set:			ypos (lparam); break;
	case	messages::color_get:		return (color ());
	case	messages::color_set:		color (lparam); break;
	default:
		debug_message ("picker::control_message: unknown - m %d, w %d, l %d.", message, wparam, lparam);
		break;
	}

	InvalidateRect (hwnd, NULL, FALSE);

	// color has changed
	notify_parent (messages::release | color ());

	return (result);
}


void
picker::visual (
	IN int value)
{
	if (visual_mode == value)
		return;

	if ((value & modes::mask) < 0 ||
		(value & modes::mask) > xgui::modes::max)
		// invalid mode
		return;

	debug_state ("mode_set = %d", value);

	// current position is not preserved
	x_pos = 0;
	y_pos = 0;
	additional_component = 0;
	visual_mode = value & modes::mask;

	// blend should be drawn, because visual mode have changed
	should_repaint = true;

	max_x = max::picker [visual_mode][1];
	max_y = max::picker [visual_mode][2];
	repaint ();
}


void
picker::component (
	IN int value)
{
	if (value < 0 || value > max::picker [visual_mode][0])
		// new value is not in range
		return;

	debug_state ("component_set = %d", value);

	additional_component = value;

	// blend should be drawn, because component has changed
	should_repaint = true;
	repaint ();
}


COLORREF
picker::color (void) const
{
	switch (visual_mode)
	{
	case	modes::rgb_red:
		return (RGB (
			additional_component * 255 / max::rgb_red,
			y_pos * 255 / max::rgb_green,
			x_pos * 255 / max::rgb_blue));

	case	modes::rgb_green:
		return (RGB (
			y_pos * 255 / max::rgb_red,
			additional_component * 255 / max::rgb_green,
			x_pos * 255 / max::rgb_blue));

	case	modes::rgb_blue:
		return (RGB (
			x_pos * 255 / max::rgb_red,
			y_pos * 255 / max::rgb_green,
			additional_component * 255 / max::rgb_blue));

	case	modes::hsv_hue:
		return (convert::hsv2rgb (
			additional_component / (double) scale::hsv_hue,
			x_pos / (double) scale::hsv_sat,
			y_pos / (double) scale::hsv_value));

	case	modes::hsv_sat:
		return (convert::hsv2rgb (
			x_pos / (double) scale::hsv_hue,
			additional_component / (double) scale::hsv_sat,
			y_pos / (double) scale::hsv_value));

	case	modes::hsv_value:
		return (convert::hsv2rgb (
			x_pos / (double) scale::hsv_hue,
			y_pos / (double) scale::hsv_sat,
			additional_component / (double) scale::hsv_value));

	case	modes::hls_hue:
		return (convert::hls2rgb (
			additional_component / (double) scale::hls_hue,
			y_pos / (double) scale::hls_light,
			x_pos / (double) scale::hls_sat));

	case	modes::hls_light:
		return (convert::hls2rgb (
			x_pos / (double) scale::hls_hue,
			additional_component / (double) scale::hls_light,
			y_pos / (double) scale::hls_sat));

	case	modes::hls_sat:
		return (convert::hls2rgb (
			x_pos / (double) scale::hls_hue,
			y_pos / (double) scale::hls_light,
			additional_component / (double) scale::hls_sat));
	}
	return (RGB (0, 0, 0));
}


void
picker::color (
	IN COLORREF value)
{
	int	x, y, comp;
	BYTE	red, green, blue;
	double	hue, light, sat, val;

	debug_state ("color_set = %x", value);

	red = GetRValue (value);
	green = GetGValue (value);
	blue = GetBValue (value);

	switch (visual_mode)
	{
	case	modes::rgb_red:
		x = blue * max::rgb_blue / 255;
		y = green * max::rgb_green / 255;
		comp = red * max::rgb_red / 255;
		break;

	case	modes::rgb_green:
		x = blue * max::rgb_blue / 255;
		y = red * max::rgb_red / 255;
		comp = green * max::rgb_green / 255;
		break;

	case	modes::rgb_blue:
		x = red * max::rgb_red / 255;
		y = green * max::rgb_green / 255;
		comp = blue * max::rgb_blue / 255;
		break;

	case	modes::hsv_hue:
		convert::rgb2hsv (
			// scale red, green and blue to [0, 1]
			red / (double) max::rgb_red,
			green / (double) max::rgb_green,
			blue / (double) max::rgb_blue,
			&hue, &sat, &val);
		x = (int) ceil (sat * scale::hsv_sat);
		y = (int) ceil (val * scale::hsv_value);
		comp = (int) ceil (hue * scale::hsv_hue);
		break;

	case	modes::hsv_sat:
		convert::rgb2hsv (
			// scale red, green and blue to [0, 1]
			red / (double) max::rgb_red,
			green / (double) max::rgb_green,
			blue / (double) max::rgb_blue,
			&hue, &sat, &val);
		x = (int) ceil (hue * scale::hsv_hue);
		y = (int) ceil (val * scale::hsv_value);
		comp = (int) ceil (sat * scale::hsv_sat);
		break;

	case	modes::hsv_value:
		convert::rgb2hsv (
			// scale red, green and blue to [0, 1]
			red / (double) max::rgb_red,
			green / (double) max::rgb_green,
			blue / (double) max::rgb_blue,
			&hue, &sat, &val);
		x = (int) ceil (hue * scale::hsv_hue);
		y = (int) ceil (sat * scale::hsv_sat);
		comp = (int) ceil (val * scale::hsv_value);
		break;

	case	modes::hls_hue:
		convert::rgb2hls (
			// scale red, green and blue to [0, 1]
			red / (double) max::rgb_red,
			green / (double) max::rgb_green,
			blue / (double) max::rgb_blue,
			&hue, &light, &sat);
		x = (int) ceil (sat * scale::hls_sat);
		y = (int) ceil (light * scale::hls_light);
		comp = (int) ceil (hue * scale::hls_hue);
		break;

	case	modes::hls_light:
		convert::rgb2hls (
			// scale red, green and blue to [0, 1]
			red / (double) max::rgb_red,
			green / (double) max::rgb_green,
			blue / (double) max::rgb_blue,
			&hue, &light, &sat);
		x = (int) ceil (hue * scale::hls_hue);
		y = (int) ceil (sat * scale::hls_sat);
		comp = (int) ceil (light * scale::hls_light);
		break;

	case	modes::hls_sat:
		convert::rgb2hls (
			// scale red, green and blue to [0, 1]
			red / (double) max::rgb_red,
			green / (double) max::rgb_green,
			blue / (double) max::rgb_blue,
			&hue, &light, &sat);
		x = (int) ceil (hue * scale::hls_hue);
		y = (int) ceil (light * scale::hls_light);
		comp = (int) ceil (sat * scale::hls_sat);
		break;

	default:
		ASSERT (FALSE);	// improper mode!
		x = 0;
		y = 0;
		comp = 0;
	}

	if (x_pos == x && y_pos == y && additional_component == comp)
		return;

	ASSERT (x >= 0 && x <= max_x);
	ASSERT (y >= 0 && y <= max_x);
	ASSERT (comp >= 0 && comp <= max::picker [visual_mode][0]);

	// set redraw blend flag if component has changed
	should_repaint |= (additional_component != comp);
	x_pos = x;
	y_pos = y;
	additional_component = comp;

	repaint ();
}


void
picker::xpos (
	IN int value)
{
	if (x_pos == value)
		return;
	x_pos = value;
	in_range (x_pos, 0, max_x);
	repaint ();
}


void
picker::ypos (
	IN int value)
{
	if (y_pos == value)
		return;
	y_pos = value;
	in_range (y_pos, 0, max_y);
	repaint ();
}


} // xgui namespace
