// xgui 0.0.4 / 2002-08-07
//	slider.cpp
//
//	http://606u.dir.bg/
//	606u@dir.bg

#include "_p.h"

#include "slider.h"

#include "resource.h"


namespace xgui {


// slider's static members initialization
bool		slider::class_registered = false;
const char	*slider::class_name = "xgui_slider";
WNDCLASS	slider::window_class = { 0 };
ATOM		slider::class_atom = 0;


slider::slider (void)
	: base ()
{
	// initialize
	visual_mode = -1;
	layout_mode = -1;
	start_val = -1;
	end_val = -1;
	cur_pos = -1;
	prev_pos = -1;
	stepsize = -1;
	pagesize = -1;
	triangle_size = -1;

	bmp_handle = NULL;
	row_buffer = NULL;
	additional_components [0] =
		additional_components [1] =
		additional_components [2] =
		additional_components [3] = -1;
}


slider::~slider ()
{
	if (bmp_handle)
		DeleteObject (bmp_handle), bmp_handle = NULL;

	if (row_buffer)
		delete [] row_buffer, row_buffer = NULL;

	// controller owns the palette
}


bool
slider::register_class (
	IN HINSTANCE library_instance)
{
	setup_wndclass (window_class, library_instance, class_name);
	return (wndclass::reg (class_name, &window_class, &class_atom));
}


bool
slider::unregister_class (
	IN HINSTANCE library_instance)
{
	return (wndclass::unreg (library_instance, class_name));
}


bool
slider::setup (
	IN HWND hwnd,
	IN LPCREATESTRUCT pcs)
{
	if (base::setup (hwnd, pcs))
	{
		// no default mode; should be set using window style
		layout (pcs->style & 0xFF00);
		visual (pcs->style & 0x00FF);

		// get triangle size from bits 9 to 12 (and add 4)
		triangle_size = ((pcs->style & 0x0F00) >> 8) + 4;

		return (size_changed (pcs->cx, pcs->cy));
	}

	return (false);
}


void
slider::mouse_move (
	IN short x,
	IN short y,
	IN int /* modifiers */)
{
	double	d;
	int		new_pos;

	SetCursor (LoadCursor (NULL, IDC_ARROW));
	if (!tracking)
		return;

	if (layout_mode & layout_horizontal)
	{
		// horizontal
		in_range (x, blend_rect.left, blend_rect.right);
		
		d = static_cast <double> (x - blend_rect.left) /
			static_cast <double> (blend_rect.right - blend_rect.left);
	}
	else // if (mode & mode_vertical)
	{
		// vertical
		in_range (y, blend_rect.top, blend_rect.bottom);

		d = static_cast <double> (y - blend_rect.top) /
			static_cast <double> (blend_rect.bottom - blend_rect.top);
	}

	new_pos = static_cast <int> ((end_val - start_val) * d +
		(end_val > start_val ? 0.5 : -0.5)) + start_val;
	if (new_pos == cur_pos)
		// don't redraw if not necessary
		return;

	cur_pos = new_pos;
	InvalidateRect (hwnd, NULL, FALSE);

	debug_state ("mouse_move, cur_pos = %d in [%d, %d]", cur_pos, start_val, end_val);

	notify_parent (messages::drag | cur_pos);
}


void
slider::mouse_up (
	IN enum mouse_buttons button,
	IN short x,
	IN short y,
	IN int modifiers)
{
	if (button == button_left && tracking)
		// position has possibly changed
		notify_parent (messages::release | cur_pos);
	base::mouse_up (button, x, y, modifiers);
}


void
slider::mouse_wheel (
	IN short delta)
{
	int		new_pos;

	// don't process if mouse is being tracked
	if (tracking)
		return;

	// delta is either positive or negative
	new_pos = cur_pos - delta / WHEEL_DELTA;

	in_range (new_pos, start_val, end_val);
	if (new_pos == cur_pos)
		// don't redraw if not necessary
		return;
	
	cur_pos = new_pos;
	notify_parent (messages::release | cur_pos);
	InvalidateRect (hwnd, NULL, FALSE);
}


void
slider::key_down (
	IN int key_code,
	IN int /* modifiers */)
{
	int		new_pos;
	int		delta;

	// don't process if mouse is being tracked
	if (tracking)
		return;

	// process scroll operation
	new_pos = cur_pos;
	delta = 0;
	switch (key_code)
	{
	case	VK_UP:
	case	VK_LEFT:
		delta = -stepsize; break;
	case	VK_DOWN:
	case	VK_RIGHT:
		delta = stepsize; break;
	case	VK_PRIOR:
		delta = -pagesize; break;
	case	VK_NEXT:
		delta = pagesize; break;
	case	VK_HOME:
		new_pos = start_val; break;
	case	VK_END:
		new_pos = end_val; break;
	default:
		return;
	}

	if (end_val > start_val)
	{
		new_pos += delta;
		in_range (new_pos, start_val, end_val);
	}
	else
	{
		new_pos -= delta;
		in_range (new_pos, end_val, start_val);
	}

	if (new_pos == cur_pos)
		// don't redraw if not necessary
		return;

	cur_pos = new_pos;
	notify_parent (messages::release | cur_pos);
	InvalidateRect (hwnd, NULL, FALSE);

	debug_state ("key_down, cur_pos = %d in [%d, %d]", cur_pos, start_val, end_val);
}


void
slider::paint (
	IN LPPAINTSTRUCT pnt)
{
	HDC		hdc;

	// create compatible DC
	hdc = CreateCompatibleDC (pnt->hdc);
	if (hdc != NULL)
	{
		HBRUSH	bg_fill;
		double	d;

		SelectObject (hdc, bmp_handle);

		// paint control's background
		bg_fill = CreateSolidBrush (GetSysColor (COLOR_BTNFACE));
		SelectObject (hdc, GetStockObject (NULL_PEN));
		SelectObject (hdc, bg_fill);
		Rectangle (hdc, 0, 0, frame.right - frame.left + 1, frame.bottom - frame.top + 1);
		SelectObject (hdc, GetStockObject (WHITE_BRUSH));
		DeleteObject (bg_fill);

		// paint blend and frame
		panel::draw (hdc, &panel_rect,
			panel::sunken | panel::blackbox);
		GdiFlush ();
		blend ();

		d = (double) (cur_pos - start_val) / (end_val - start_val);

		if (layout_mode & layout_horizontal)
		{
			// horizontal slider
			int		x;

			x = (int) ((blend_rect.right - blend_rect.left) * d);

			if (layout_mode & layout_tr_top)
				triangle::draw (hdc, blend_rect.left + x, panel_rect.top - 2,
					triangle::bottom, triangle_size,
					GetSysColor (COLOR_WINDOWTEXT), GetSysColor (COLOR_WINDOW));
			if (layout_mode & layout_tr_bottom)
				triangle::draw (hdc, blend_rect.left + x, panel_rect.bottom + 2,
					triangle::top, triangle_size,
					GetSysColor (COLOR_WINDOWTEXT), GetSysColor (COLOR_WINDOW));
		}
		else // if (layout_mode & layout_vertical)
		{
			// vertical slider
			int	y;

			y = (int) ((blend_rect.bottom - blend_rect.top) * d);

			if (layout_mode & layout_tr_left)
				triangle::draw (hdc, panel_rect.left - 2, blend_rect.top + y,
					triangle::right, triangle_size,
					GetSysColor (COLOR_WINDOWTEXT), GetSysColor (COLOR_WINDOW));
			if (layout_mode & layout_tr_right)
				triangle::draw (hdc, panel_rect.right + 2, blend_rect.top + y,
					triangle::left, triangle_size,
					GetSysColor (COLOR_WINDOWTEXT), GetSysColor (COLOR_WINDOW));
		}

		prev_pos = cur_pos;

		// paint a focus rectangle if necessary
		if (GetFocus () == hwnd)
			DrawFocusRect (hdc, &frame);

		// bitblt and delete cache DC
		GdiFlush ();
		BitBlt (pnt->hdc, 0, 0,
			frame.right - frame.left, frame.bottom - frame.top, hdc, 0, 0, SRCCOPY);
		DeleteDC (hdc);
	}
}


bool
slider::size_changed (
	IN int hsize,
	IN int vsize)
{
	if (!base::size_changed (hsize, vsize))
		return (false);

	CopyRect (&panel_rect, &interaction_area);
	if (layout_mode & layout_horizontal)
	{
		// horizontal slider
		if (layout_mode & layout_tr_top)
			panel_rect.top += (triangle_size + 1);
		if (layout_mode & layout_tr_bottom)
			panel_rect.bottom -= (triangle_size + 1);
		InflateRect (&panel_rect, -(triangle_size - 3), 0);
	}
	else // if (layout_mode & layout_vertical)
	{
		// vertical slider
		if (layout_mode & layout_tr_left)
			panel_rect.left += (triangle_size + 1);
		if (layout_mode & layout_tr_right)
			panel_rect.right -= (triangle_size + 1);
		InflateRect (&panel_rect, 0, -(triangle_size - 3));
	}

	CopyRect (&blend_rect, &panel_rect);
	InflateRect (&blend_rect, -2, -2);

	if (layout_mode & layout_horizontal)
		row_buff_size = blend_rect.right - blend_rect.left + 1;
	else
		row_buff_size = blend_rect.bottom - blend_rect.top + 1;
	if (row_buffer)
		delete [] row_buffer;
	row_buffer = new DWORD [row_buff_size];
	if (row_buffer == NULL)
		return (false);

	// create cache bitmap
	bmp_info.bmiHeader.biSize = sizeof (BITMAPINFO);
	bmp_info.bmiHeader.biWidth = hsize;
	bmp_info.bmiHeader.biHeight = vsize;
	bmp_info.bmiHeader.biPlanes = 1;
	bmp_info.bmiHeader.biBitCount = 32;
	bmp_info.bmiHeader.biCompression = BI_RGB;
	bmp_info.bmiHeader.biSizeImage = hsize * vsize * 32 / 8;
	bmp_info.bmiHeader.biXPelsPerMeter =
		bmp_info.bmiHeader.biYPelsPerMeter = 72 * 2 * 1000;
	bmp_info.bmiHeader.biClrUsed = 0;
	bmp_info.bmiHeader.biClrImportant = 0;
	bmp_data = NULL;
	bmp_handle = CreateDIBSection (
		NULL, &bmp_info, DIB_RGB_COLORS, (void **) &bmp_data, NULL, 0);
	if (bmp_handle == NULL)
	{
		debug_message ("slider::size_changed: CreateDIBSection failed.");
		return (false);
	}
	return (true);
}


LRESULT
slider::control_message (
	IN enum control_codes message,
	IN WPARAM wparam,
	IN LPARAM lparam)
{
	LRESULT		result;

	// ask base class first
	result = base::control_message (message, wparam, lparam);
	if (result)
		return (result);

	result = 0;
	switch (message)
	{
	case	messages::visual_get:	return (visual ());
	case	messages::visual_set:	visual (lparam); break;
	case	messages::layout_get:	return (layout ());
	case	messages::layout_set:	layout (lparam); break;
	case	messages::start_get:	return (start ());
	case	messages::start_set:	start (lparam); break;
	case	messages::end_get:		return (end ());
	case	messages::end_set:		end (lparam); break;
	case	messages::pos_get:		return (position ());
	case	messages::pos_set:		position (lparam); break;
	case	messages::component_get:	return (component (wparam));
	case	messages::component_set:	component (wparam, lparam); break;
	default:
		debug_message ("slider::control_message: unknown - m %d, w %d, l %d.", message, wparam, lparam);
		break;
	}
	InvalidateRect (hwnd, NULL, FALSE);
	return (result);
}


void
slider::layout (
	IN int value)
{
	layout_mode = value;
	repaint ();
}


void
slider::visual (
	IN int value)
{
	if ((value & modes::mask) < 0 ||
		(value & modes::mask) > xgui::modes::max)
		// invalid mode
		return;

	visual_mode = value;
	switch (visual_mode & modes::mask)
	{
	case	modes::rgb_red:
	case	modes::rgb_green:
	case	modes::rgb_blue:
		start_val = 0;
		end_val = max::rgb;
		stepsize = scale::rgb;
		pagesize = scale::rgb * 10;
		break;

	case	modes::hsv_hue:
		start_val = 0;
		end_val = max::hsv_hue;
		stepsize = scale::hsv_hue;
		pagesize = scale::hsv_hue * 10;
		break;

	case	modes::hsv_sat:
		start_val = 0;
		end_val = max::hsv_sat;
		stepsize = scale::hsv_sat / 100;
		pagesize = scale::hsv_sat / 10;
		break;

	case	modes::hsv_value:
		start_val = 0;
		end_val = max::hsv_value;
		stepsize = scale::hsv_value / 100;
		pagesize = scale::hsv_value / 10;
		break;

	case	modes::hls_hue:
		start_val = 0;
		end_val = max::hls_hue;
		stepsize = scale::hls_hue;
		pagesize = scale::hls_hue * 10;
		break;

	case	modes::hls_light:
		start_val = 0;
		end_val = max::hls_light;
		stepsize = scale::hls_light / 100;
		pagesize = scale::hls_light / 10;
		break;

	case	modes::hls_sat:
		start_val = 0;
		end_val = max::hls_sat;
		stepsize = scale::hls_sat / 100;
		pagesize = scale::hls_sat / 10;
		break;
	}

	// position is not preserved
	prev_pos = -9999;
	cur_pos = (visual_mode & modes::reverse ? end_val : start_val);

	// reinitialize components
	additional_components [0] =
		additional_components [1] =
		additional_components [2] =
		additional_components [3] = 0;

	repaint ();
}


void
slider::start (
	IN int value)
{
	if (start_val == value)
		return;
	start_val = value;
	in_between (cur_pos, start_val, end_val);
	repaint ();
}


void
slider::end (
	IN int value)
{
	if (end_val == value)
		return;
	end_val = value;
	in_between (cur_pos, start_val, end_val);
	repaint ();
}


void
slider::position (
	IN int value)
{
	if (visual_mode & modes::reverse)
		// invert if neccessary
		value = end_val - value;
	if (cur_pos == value)
		return;
	cur_pos = value;
	in_between (cur_pos, start_val, end_val);
	repaint ();
}


void
slider::component (
	IN int index,
	IN int value)
{
	if (index < 0 || index > 1)
		// invalid index
		return;
	if (additional_components [index] == value)
		return;
	if (value < 0 ||
		value > max::slider [visual_mode & modes::mask][index + 1])
		// value out of range
		return;
	additional_components [index] = value;
	repaint ();
}


} // xgui namespace
