// xgui 0.0.4 / 2002-08-07
//	xgui_statics.cpp
//
//	http://606u.dir.bg/
//	606u@dir.bg

#include "_p.h"

#include "slider.h"
#include "picker.h"
#include "well.h"
#include "controller.h"



namespace xgui {


namespace panel {

void
draw (
	IN HDC hdc,
	IN int left,
	IN int top,
	IN int right,
	IN int bottom,
	IN int flags)
{
	HPEN	highlight_pen, shadow_pen, old_pen;
	HBRUSH	old_brush;

	shadow_pen = CreatePen (PS_SOLID, 0, GetSysColor (COLOR_BTNSHADOW));
	highlight_pen = CreatePen (PS_SOLID, 0, GetSysColor (COLOR_BTNHILIGHT));

	// old pen is preserved here
	if (flags & panel::sunken)
		old_pen = (HPEN) SelectObject (hdc, shadow_pen);
	else
		old_pen = (HPEN) SelectObject (hdc, highlight_pen);

	// draw left and top side
	MoveToEx (hdc, left, bottom, NULL);
	LineTo (hdc, left, top);
	LineTo (hdc, right + 1, top);

	if (flags & panel::sunken)
		SelectObject (hdc, highlight_pen);
	else
		SelectObject (hdc, shadow_pen);

	// bottom and right side
	MoveToEx (hdc, left + 1, bottom, NULL);
	LineTo (hdc, right, bottom);
	LineTo (hdc, right, top);

	SelectObject (hdc, GetStockObject (BLACK_PEN));

	if (flags & panel::blackbox)
	{
		old_brush = (HBRUSH) SelectObject (hdc, GetStockObject (NULL_BRUSH));
		Rectangle (hdc, left + 1, top + 1,
			right, bottom);
		SelectObject (hdc, old_brush);
	}

	// restore the old pen here
	SelectObject (hdc, old_pen);

	DeleteObject (shadow_pen);
	DeleteObject (highlight_pen);
}

} // panel namespace


namespace triangle {

void
draw (
	IN HDC hdc,
	IN int x,
	IN int y,
	IN enum types type,
	IN int size,
	IN COLORREF line_color,
	IN COLORREF fill_color)
{
	POINT	pt [3];								// array of points used for polygon
	HPEN	stroke_pen, old_pen;
	HBRUSH	fill_brush, old_brush;

	size --;
	// fill the array with points according to triangle mode
	pt [0].x = x; pt [0].y = y;
	switch (type)
	{
	case	left:
		pt [1].x = x + size; pt [1].y = y + size;
		pt [2].x = x + size; pt [2].y = y - size;
		break;

	case	top:
		pt [1].x = x + size; pt [1].y = y + size;
		pt [2].x = x - size; pt [2].y = y + size;
		break;

	case	right:
		pt [1].x = x - size; pt [1].y = y - size;
		pt [2].x = x - size; pt [2].y = y + size;
		break;
	
	case	bottom:
		pt [1].x = x - size; pt [1].y = y - size;
		pt [2].x = x + size; pt [2].y = y - size;
		break;

	case	left_top:
		pt [1].x = x + size; pt [1].y = y;
		pt [2].x = x; pt [2].y = y + size;
		break;

	case	left_bottom:
		pt [1].x = x + size; pt [1].y = y;
		pt [2].x = x; pt [2].y = y - size;
		break;

	case	right_top:
		pt [1].x = x - size; pt [1].y = y;
		pt [2].x = x; pt [2].y = y + size;
		break;

	case	right_bottom:
		pt [1].x = x - size; pt [1].y = y;
		pt [2].x = x; pt [2].y = y - size;
		break;

	default:
		// unknown triangle orientation
		debug_message ("base::draw_triangle: unknown type %d", type);
		ASSERT (FALSE);
		return;
	}

	stroke_pen = CreatePen (PS_SOLID, 0, line_color);
	fill_brush = CreateSolidBrush (fill_color);

	old_pen = (HPEN) SelectObject (hdc, stroke_pen);
	old_brush = (HBRUSH) SelectObject (hdc, fill_brush);

	Polygon (hdc, pt, 3);
	
	SelectObject (hdc, old_pen);
	SelectObject (hdc, old_brush);

	DeleteObject (stroke_pen);
	DeleteObject (fill_brush);
}

} // triangle namespace


namespace box {

void
draw (
	IN HDC hdc,
	IN int left,
	IN int top,
	IN int side,
	IN COLORREF line_color,
	IN COLORREF fill_color)
{
	HPEN	stroke_pen, old_pen;
	HBRUSH	fill_brush, old_brush;

	stroke_pen = CreatePen (PS_SOLID, 0, line_color);
	fill_brush = CreateSolidBrush (fill_color);

	old_pen = (HPEN) SelectObject (hdc, stroke_pen);
	old_brush = (HBRUSH) SelectObject (hdc, fill_brush);

	Rectangle (hdc, left, top, left + side, top + side);
	
	SelectObject (hdc, old_pen);
	SelectObject (hdc, old_brush);

	DeleteObject (stroke_pen);
	DeleteObject (fill_brush);
}

} // box namespace


namespace convert {

int
hsv2rgb (
	IN double hue,
	IN double sat,
	IN double value,
	OUT double *red,
	OUT double *green,
	OUT double *blue)
{
	double	frac, coef1, coef2, coef3;
	double	intp;
	// hsv values valid?
	if (sat < 0.0 || sat > 1.0 || value < 0.0 || value > 1.0) return (-1);
	if (hue < 0.0 || hue > 360.0) return (-1);

	// gray?
	if (sat == 0.0)								
		*red = *green = *blue = value;
	else
	{
		// hue (chromatic) 360 == hue 0
		if (hue == 360.0) hue = 0;
		hue = hue / 60; 						// hue in [0, 6)
		frac = modf (hue, &intp);				// split hue to integer and fraction
		coef1 = value * (1 - sat);
		coef2 = value * (1 - sat * frac);
		coef3 = value * (1 - sat * (1 - frac));
		switch ((int) intp)
		{
		case 0:	*red = value; *green = coef3; *blue = coef1; break;
		case 1:	*red = coef2; *green = value; *blue = coef1; break;
		case 2:	*red = coef1; *green = value; *blue = coef3; break;
		case 3:	*red = coef1; *green = coef2; *blue = value; break;
		case 4:	*red = coef3; *green = coef1; *blue = value; break;
		case 5:	*red = value; *green = coef1; *blue = coef2; break;
		}
	}
	return (0);
}


int
rgb2hsv (
	IN double red,
	IN double green,
	IN double blue,
	OUT double *hue,
	OUT double *sat,
	OUT double *value)
{
	double max, min, delta;

	max = MAX (red, MAX (green, blue));
	min = MIN (red, MIN (green, blue));
   
	// check the rgb values to see if valid
	if (min < 0.0 || max > 1.0) return (-1); 	// out of range
	
	*value = max;								// calculate the value v

	if (max > 0.0)
		*sat = (max - min) / max;
	else
		// red = green = blue = 0
		*sat = 0.0;

	if (*sat == 0.0)
		*hue = 0.0;
	else
	{
		delta = max - min;
		if (red == max)
			// between yellow and magenta
			*hue = (green - blue) / delta;
		else if (green  == max)
			// between cyan and yellow
			*hue = 2 + (blue - red) / delta;
		else 
			// between magenta and cyan
			*hue = 4 + (red - green) / delta;

		// hue to degrees
		*hue = *hue * 60;
		if (*hue < 0)
			// make hue > 0
			*hue = *hue + 360;
	}
	
	ASSERT (*hue >= 0.0 && *hue <= 360.0);
	ASSERT (*sat >= 0.0 && *sat <= 1.0);
	ASSERT (*value >= 0.0 && *value <= 1.0);

	return (0); 								
}


// inline is put here instead of header file, because it should be inline
// for current file only - for hls_light & hls_sat blends mostly
inline double
hls_value (
	IN double m1,
	IN double m2,
	IN double hue)
{
    if (hue > 360.0) hue -= 360.0;
    if (hue < 0.0) hue += 360.0;
	if (hue <= 60.0)
		return (m1 + (m2 - m1) * hue / 60.0);
	else if (hue <= 180.0)
		return (m2);
	else if (hue <= 240.0)
		return (m1 + (m2 - m1) * (240.0 - hue) / 60.0);
	else 
		return (m1);
}


int
hls2rgb (
	IN double hue,
	IN double light,
	IN double sat,
	OUT double *red,
	OUT double *green,
	OUT double *blue)
{
	double m1, m2;

	// hls values valid?
	if (sat < 0.0 || sat > 1.0 || light < 0.0 || light > 1.0) return (-1);
	if (hue < 0.0 || hue > 360.0) return (-1);

	if (light <= 0.5) 
		m2 = light * (1.0 + sat);
	else  
		m2 = light + sat - light * sat;

	m1 = 2 * light - m2;

	if (sat == 0.0)
		// gray
		*red = *green = *blue = light;
	else
	{
		*red = hls_value (m1, m2, hue + 120.0);
		*green = hls_value (m1, m2, hue);
		*blue = hls_value (m1, m2, hue - 120.0);
	}
	return (0);
}


int
rgb2hls (
	IN double red,
	IN double green,
	IN double blue,
	OUT double *hue,
	OUT double *light,
	OUT double *sat)
{
	double	max, min, delta;

	max = MAX (red, MAX (green, blue));
	min = MIN (red, MIN (green, blue));

	// rgb values valid?
	if (min < 0.0 || max > 1.0) return (-1);

	*light = (max + min) / 2.0;

	if (max == min)
	{
		// achomatic case: r = g = b
		*sat = 0.0;
		*hue = 0.0;
	}
	else
	{
		// chromatic case

		if (*light <= 0.5)
			*sat = (max - min) / (max + min);
		else
			*sat = (max - min) / (2 - max - min);

		delta = max - min;
		if (red == max)
			// between yellow and magenta
			*hue = (green - blue) / delta;
		else if (green == max)
			// between cyan and yellow
			*hue = 2 + (blue - red) / delta;
		else
			// between magenta and cyan
			*hue = 4 + (red - green) / delta;

		// convert hue to degrees (positive)
		*hue *= 60;
		while (*hue < 0.0)
			*hue += 360;
	}
	return (0); 								
}

} // convert namespace


namespace blend {

void
rgb (
	OUT DWORD *buffer,
	IN int samples,
	IN COLORREF start,
	IN COLORREF end,
	IN OPTIONAL bool use_fast_blends)	// = FAST_BLENDS
{
	if (use_fast_blends)
	{
		// all are scaled (means, they are shifted left "int_extend" number of bits)
		//	to use integer math in the sake of performance
		int	red, green, blue;
		int	red_adv, green_adv, blue_adv;

		// positive advances are incremented with 1 to compensate rounding errors
		red = convert::scaled_red (start);
		green = convert::scaled_green (start);
		blue = convert::scaled_blue (start);
		red_adv = (convert::scaled_red (end) - red) / (samples - 1);
		red_adv += (red_adv > 0 ? 1 : 0);
		green_adv = (convert::scaled_green (end) - green) / (samples - 1);
		green_adv += (green_adv > 0 ? 1 : 0);
		blue_adv = (convert::scaled_blue (end) - blue) / (samples - 1);
		blue_adv += (blue_adv > 0 ? 1 : 0);

		while (samples--)
		{
			// set current pixel (in DIB bitmap format is BGR, not RGB!)
			*buffer++ = convert::scaled2bgr (red, green, blue);

			// advance color values to the next pixel
			red += red_adv;
			green += green_adv;
			blue += blue_adv;
		}
	}
	else
	{
		BYTE	r1, g1, b1, r2, g2, b2;
		r1 = GetRValue (start);
		g1 = GetGValue (start);
		b1 = GetBValue (start);
		r2 = GetRValue (end);
		g2 = GetGValue (end);
		b2 = GetBValue (end);

		int	i = 0;
		while (i < samples)
		{
			BYTE	r, g, b;
			r = (BYTE) (r1 + (r2 - r1) * i / (samples - 1));
			g = (BYTE) (g1 + (g2 - g1) * i / (samples - 1));
			b = (BYTE) (b1 + (b2 - b1) * i / (samples - 1));
			*buffer++ = RGB (b, g, r);
			++i;
		}
	}
}


void
hsv_hue (
	OUT DWORD *buffer,
	IN int samples,
	IN double sat,
	IN double val_fp,
	IN OPTIONAL bool use_fast_blends)	// = FAST_BLENDS
{
	// use fast blends? (defined in xgui_statics.h)
	if (use_fast_blends)
	{
		// value, but as integer in [0, 255 << int_extend]
		int		val;

		// loop counter
		int		j;

		// coefficients and advances
		int		coef1, coef2, coef3;
		int		coef2_adv, coef3_adv;

		// current position and advance to the next one
		double	pos, pos_adv;

		//
		// hue increments in [0, 360); indirectly
		//	intp changes - 0, 1, 2, 3, 4, 5; indirectly (separate loops)
		//	frac increments in [0, 1) six times; indirectly (coefficients)
		// sat - const, in [0, 1]
		// val - const, in [0, (255 << int_extend)]
		//
		// coef1 => val * (1 - sat)              => const, = val * (1 - sat)
		// coef2 => val * (1 - sat * frac)       => changes from val to val * (1 - sat)
		// coef3 => val * (1 - sat * (1 - frac)) => changes from val * (1 - sat) to val
		//

		// constants
		val = to_int (val_fp * 255.0 * int_extend_multiply);
		coef1 = to_int (val * (1 - sat));

		// prepare
		pos = 0;
		pos_adv = (double) samples / 6.0;

		// hue in [0, 60)
		pos += pos_adv;
		j = (int) pos;
		coef3 = coef1;
		coef3_adv = (val - coef3) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (val, coef3, coef1), coef3 += coef3_adv;
		coef3 = val;

		// hue in [60, 120)
		pos += pos_adv;
		j = (int) pos - (int) (1 * pos_adv);
		coef2 = val;
		coef2_adv = (coef1 - coef2) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (coef2, val, coef1), coef2 += coef2_adv;
		coef2 = coef1;

		// hue in [120, 180)
		pos += pos_adv;
		j = (int) pos - (int) (2 * pos_adv);
		coef3 = coef1;
		coef3_adv = (val - coef3) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (coef1, val, coef3), coef3 += coef3_adv;
		coef3 = val;

		// hue in [180, 240)
		pos += pos_adv;
		j = (int) pos - (int) (3 * pos_adv);
		coef2 = val;
		coef2_adv = (coef1 - coef2) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (coef1, coef2, val), coef2 += coef2_adv;
		coef2 = coef1;

		// hue in [240, 300)
		pos += pos_adv;
		j = (int) pos - (int) (4 * pos_adv);
		coef3 = coef1;
		coef3_adv = (val - coef3) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (coef3, coef1, val), coef3 += coef3_adv;
		coef3 = val;

		// hue in [300, 360)
		pos += (pos_adv + 0.1);	// + 0.1 because of floating-point math's rounding errors
		j = (int) pos - (int) (5 * pos_adv);
		coef2 = val;
		coef2_adv = (coef1 - coef2) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (val, coef1, coef2), coef2 += coef2_adv;
	}
	
	else
	{
		double	hue, hue_adv;

		hue = 0.0;
		hue_adv = 360.0 / samples;
		while (samples--)
			*buffer++ = convert::bgr (convert::hsv2rgb (hue, sat, val_fp)),
			hue += hue_adv;
	}
}


void
hsv_sat (
	OUT DWORD *buffer,
	IN int samples,
	IN double hue,
	IN double val_fp,
	IN OPTIONAL bool use_fast_blends)	// = FAST_BLENDS
{
	// use fast blends? (defined in xgui_statics.h)
	if (use_fast_blends)
	{
		// value, but as integer in [0, 255 << int_extend]
		int		val;

		// loop counter
		int		j;

		// coefficients and advances
		signed int		coef1, coef2, coef3;
		signed int		coef1_adv, coef2_adv, coef3_adv;

		double	intp, frac;

		//
		// hue - const, in [0, 360)
		//	intp - const in 0, 1, 2, 3, 4, 5
		//	frac - const in [0, 1)
		// sat - increments, in [0, 1]; indirectly (coefficients)
		// val - const, in [0, (255 << int_extend)]
		//
		// coef1 => val * (1 - sat)              => changes from val to 0
		// coef2 => val * (1 - sat * frac)       => changes from val to val * (1 - frac)
		// coef3 => val * (1 - sat * (1 - frac)) => changes from val to val * frac
		//

		// constants
		val = to_int (val_fp * 255 * int_extend_multiply);
		frac = modf (hue / 60.0, &intp);

		// prepare
		j = samples;

		coef1 = val;
		coef1_adv = -coef1 / samples;
		coef2 = val;
		coef2_adv = (int) ((1 - frac) * val - coef2) / samples;
		coef3 = val;
		coef3_adv = (int) (frac * val - coef3) / samples;

		switch ((int) intp)
		{
		case	0:
			while (j--)
				*buffer++ = convert::scaled2bgr (val, coef3, coef1),
					coef1 += coef1_adv, coef3 += coef3_adv;
			break;

		case	1:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef2, val, coef1),
					coef1 += coef1_adv, coef2 += coef2_adv;
			break;
		
		case	2:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef1, val, coef3),
					coef1 += coef1_adv, coef3 += coef3_adv;
			break;
		
		case	3:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef1, coef2, val),
					coef1 += coef1_adv, coef2 += coef2_adv;
			break;
		
		case	4:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef3, coef1, val),
					coef1 += coef1_adv, coef3 += coef3_adv;
			break;
		
		case	5:
			while (j--)
				*buffer++ = convert::scaled2bgr (val, coef1, coef2),
					coef1 += coef1_adv, coef2 += coef2_adv;
			break;
		}
	}

	else
	{
		double	sat, sat_adv;

		sat = 0.0;
		sat_adv = 1.0 / samples;
		while (samples--)
			*buffer++ = convert::bgr (convert::hsv2rgb (hue, sat, val_fp)),
			sat += sat_adv;
	}
}


void
hsv_val (
	OUT DWORD *buffer,
	IN int samples,
	IN double hue,
	IN double sat,
	IN OPTIONAL bool use_fast_blends)	// = FAST_BLENDS
{
	// use fast blends? (defined in xgui_statics.h)
	if (use_fast_blends)
	{
		// loop counter
		int		j;

		// coefficients and advances
		signed int		coef1, coef2, coef3;
		signed int		coef1_adv, coef2_adv, coef3_adv;

		int		val, val_adv;
		int		val_max;

		double	intp, frac;

		//
		// hue - const, in [0, 360)
		//	intp - const in 0, 1, 2, 3, 4, 5
		//	frac - const in [0, 1)
		// sat - const, in [0, 1]
		// val - increments, in [0, (255 << int_extend)]; indirectly (coefficients)
		//
		// coef1 => val * (1 - sat)              => changes from 0 to val * (1 - sat)
		// coef2 => val * (1 - sat * frac)       => changes from 0 to val * (1 - sat * frac)
		// coef3 => val * (1 - sat * (1 - frac)) => changes from 0 to val * (1 - sat * (1 - frac))
		//

		// constants
		frac = modf (hue / 60.0, &intp);
		val_max = 255 << int_extend;

		// prepare
		j = samples;

		coef1 = 0;
		coef1_adv = (int) (val_max * (1 - sat)) / samples;
		coef2 = 0;
		coef2_adv = (int) (val_max * (1 - sat * frac)) / samples;
		coef3 = 0;
		coef3_adv = (int) (val_max * (1 - sat * (1 - frac))) / samples;
		val = 0;
		val_adv = val_max / samples;

		switch ((int) intp)
		{
		case	0:
			while (j--)
				*buffer++ = convert::scaled2bgr (val, coef3, coef1),
					coef1 += coef1_adv, coef3 += coef3_adv, val += val_adv;
			break;

		case	1:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef2, val, coef1),
					coef1 += coef1_adv, coef2 += coef2_adv, val += val_adv;
			break;

		case	2:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef1, val, coef3),
					coef1 += coef1_adv, coef3 += coef3_adv, val += val_adv;
			break;

		case	3:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef1, coef2, val),
					coef1 += coef1_adv, coef2 += coef2_adv, val += val_adv;
			break;

		case	4:
			while (j--)
				*buffer++ = convert::scaled2bgr (coef3, coef1, val),
					coef1 += coef1_adv, coef3 += coef3_adv, val += val_adv;
			break;

		case	5:
			while (j--)
				*buffer++ = convert::scaled2bgr (val, coef1, coef2),
					coef1 += coef1_adv, coef2 += coef2_adv, val += val_adv;
			break;
		}
	}

	else
	{
		double	val, val_adv;

		val = 0.0;
		val_adv = 1.0 / samples;
		while (samples--)
			*buffer++ = convert::bgr (convert::hsv2rgb (hue, sat, val)),
			val += val_adv;
	}
}


void
hls_hue (
	OUT DWORD *buffer,
	IN int samples,
	IN double light,
	IN double sat,
	IN OPTIONAL bool use_fast_blends)	// = FAST_BLENDS
{
	// use fast blends? (defined in xgui_statics.h)
	if (use_fast_blends)
	{
		double m1, m2;

		int	m1_scaled, m2_scaled;

		// current position and advance to the next one
		double	pos, pos_adv;

		// loop counter
		int	j;

		int	red, green, blue, adv;

		//
		// red is @ hue + 120, green is @ hue, blue is @ hue - 120
		//
		//	red = value (m1, m2, hue + 120.0);
		//	green = value (m1, m2, hue);
		//	blue = value (m1, m2, hue - 120.0);
		//
		// or red (when hue = 0) = green (when hue = 120) = blue (when hue = 240)
		// which means, that we could update 3 components of 3 different pixels
		// on each iteration
		//
		//		hue		0	->	60	->	120	->	180	->	240	->	300	->	360
		//	red			1		1		2		2		3		3
		//	green		3		3		1		1		2		2
		//	blue		2		2		3		3		1		1
		//
		// unfortunatelly, this can only be done when total width is divideable on 3
		//
		// so here goes the real math
		//
		// hue changes in [0, 360) (indirectly)
		//	intp changes 0 to 3 (0: hue <= 60, 1: <= 180, 2: <= 240, 3: > 240) 4 separate blocks
		//	fracp goes from 0 to 1 indirectly (coefficients) 4 times
		// light is fixed in [0, 1]
		// sat is fixed in [0, 1]
		//
		//	red =   value (m1, m2, hue + 120.0);
		//	green = value (m1, m2, hue);
		//	blue =  value (m1, m2, hue - 120.0);
		//
		//		hue						value
		//	  0 - 60		m1 + (m2 - m1) * hue / 60.0;	(m1 -> m2)
		//	 60 - 180		m2
		//	180 - 240		m1 + (m2 - m1) * (240.0 - hue) / 60.0;	(m2 -> m1)
		//	240 - 360		m1
		//
		//	hue				0		60		120		180		240		300		360
		//	red				m2		m2	->	m1		m1		m1	->	m2		m2
		//	green			m1	->	m2		m2		m2	->	m1		m1		m1
		//	blue		->	m1		m1		m1	->	m2		m2		m2	->	m1
		//

		// calc both coefficients
		if (light <= 0.5) 
			m2 = light * (1.0 + sat);
		else  
			m2 = light + sat - light * sat;
		m1 = 2 * light - m2;
		m1_scaled = to_int (m1 * 255.0 * int_extend_multiply);
		m2_scaled = to_int (m2 * 255.0 * int_extend_multiply);

		// prepare
		pos = 0.0;
		pos_adv = (double) samples / 6.0;

		// initialize as shown above, when hue = 0
		red = m2_scaled;
		green = m1_scaled;
		blue = m1_scaled;

		// hue in [0, 60)
		pos += pos_adv;
		j = (int) pos;
		adv = (m2_scaled - m1_scaled) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (red, green, blue), green += adv;
		green = m2_scaled;

		// hue in [60, 120)
		pos += pos_adv;
		j = (int) pos - (int) (1 * pos_adv);
		adv = (m1_scaled - m2_scaled) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (red, green, blue), red += adv;
		red = m1_scaled;

		// hue in [120, 180)
		pos += pos_adv;
		j = (int) pos - (int) (2 * pos_adv);
		adv = (m2_scaled - m1_scaled) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (red, green, blue), blue += adv;
		blue = m2_scaled;

		// hue in [180, 240)
		pos += pos_adv;
		j = (int) pos - (int) (3 * pos_adv);
		adv = (m1_scaled - m2_scaled) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (red, green, blue), green += adv;
		green = m1_scaled;

		// hue in [240, 300)
		pos += pos_adv;
		j = (int) pos - (int) (4 * pos_adv);
		adv = (m2_scaled - m1_scaled) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (red, green, blue), red += adv;
		red = m2_scaled;

		// hue in [300, 360)
		pos += (pos_adv + 0.1);	// + 0.1 because of floating-point math's rounding errors
		j = (int) pos - (int) (5 * pos_adv);
		adv = (m1_scaled - m2_scaled) / j;
		while (j--)
			*buffer++ = convert::scaled2bgr (red, green, blue), blue += adv;

		//
		// BTW, this could be optimized even better, because in each step
		// only one of the components is changed - others are not needed
		// to be scaled-out to be scaled-in later for each step (const)
		// 
		// actually, the compiler will make this optimization
		//
	}

	else
	{
		double hue, hue_adv;

		hue = 0.0;
		hue_adv = 360.0 / samples;
		while (samples--)
			*buffer++ = convert::bgr (convert::hls2rgb (hue, light, sat)),
			hue += hue_adv;
	}
}


void
hls_light (
	OUT DWORD *buffer,
	IN int samples,
	IN double hue,
	IN double sat,
	IN OPTIONAL bool use_fast_blends)	// = FAST_BLENDS
{
	// use fast blends? (defined in xgui_statics.h)
	if (use_fast_blends)
	{
		int	red, green, blue;
		int	red_adv, green_adv, blue_adv;
		int	i;

		// m*'s are scaled; iter is in [0, 1] - 2 steps - light in [0, 0.5) and [0.5, 1)
		int	m1_start, m2_start, m1_end, m2_end;
		int iter;

		//
		// light changes on two steps: from 0.0 to 0.5 and from 0.5 to 1.0
		//
		// light		0.0			0.5			1.0
		//	m1			0.0		0.5-sat/2		1.0
		//	m2			0.0		0.5+sat/2		1.0
		//

		iter = 0;
		red = green = blue = 0;
		red_adv = green_adv = blue_adv = 0;
		do
		{
			if (iter == 0)
			{
				// light in [0, 0.5)
				i = samples / 2;
				m1_start = m2_start = 0;
				m1_end = to_int ((0.5 - sat / 2) * 255.0 * int_extend_multiply);
				m2_end = to_int ((0.5 + sat / 2) * 255.0 * int_extend_multiply);
			}
			else
			{
				// light in [0.5, 1.0)
				i = samples - samples / 2;	// because of rounding
				m1_start = to_int ((0.5 - sat / 2) * 255.0 * int_extend_multiply);
				m2_start = to_int ((0.5 + sat / 2) * 255.0 * int_extend_multiply);
				m1_end = m2_end = 255 << int_extend;
			}

			// advance = (end - start) / steps
			red_adv = (int) (convert::hls_value (m1_end, m2_end, hue + 120) - red) / i;
			green_adv = (int) (convert::hls_value (m1_end, m2_end, hue) - green) / i;
			blue_adv = (int) (convert::hls_value (m1_end, m2_end, hue - 120) - blue) / i;

			// this is not very fast, since there are 3 additions
			//	and 3 shifts involved not including
			//	"and"s and "or"s to build-up RGB triple
			// actually in any case only two of those are changed
			//	(one of red_adv, green_adv or blue_adv will be close to 0)
			//	so it could be split in switch with 3 cases
			while (i--)
			{
				*buffer++ = convert::scaled2bgr (red, green, blue);
				red += red_adv;
				green += green_adv;
				blue += blue_adv;
			}

			iter++;
		}
		while (iter < 2);
	}

	else
	{
		double	light, light_adv;

		light = 0.0;
		light_adv = 1.0 / samples;
		while (samples--)
			*buffer++ = convert::bgr (convert::hls2rgb (hue, light, sat)),
			light += light_adv;
	}
}


void
hls_sat (
	OUT DWORD *buffer,
	IN int samples,
	IN double hue,
	IN double light,
	IN OPTIONAL bool use_fast_blends)	// = FAST_BLENDS
{
	// use fast blends? (defined in xgui_statics.h)
	if (use_fast_blends)
	{
		int	red, green, blue;
		int	red_adv, green_adv, blue_adv;
		int	i;

		// m*'s are scaled
		int	m1_start, m2_start, m1_end, m2_end;

		//
		// sat changes from 0.0 to 1.0
		//
		//                          sat 0.0      1.0
		// light < 0.5: m2 changes from light to 2 * light
		//				m1 changes from light to 0.0
		// light > 0.5: m2 changes from light to 1.0
		//				m1 changes from light to 2 * light - 1.0
		//

		i = samples;
		m1_start = m2_start = to_int (light * 255.0 * int_extend_multiply);

		if (light <= 0.5)
		{
			// light in [0, 0.5)
			m1_end = 0;
			m2_end = to_int (2 * light * 255.0 * int_extend_multiply);
		}
		else
		{
			// light in [0.5, 1.0)
			m1_end = to_int ((2 * light - 1) * 255.0 * int_extend_multiply);
			m2_end = to_int (1.0 * 255.0 * int_extend_multiply);
		}

		red = (int) convert::hls_value (m1_start, m2_start, hue + 120);
		green = (int) convert::hls_value (m1_start, m2_start, hue);
		blue = (int) convert::hls_value (m1_start, m2_start, hue - 120);

		// advance = (end - start) / steps
		red_adv = (int) (convert::hls_value (m1_end, m2_end, hue + 120) - red) / i;
		green_adv = (int) (convert::hls_value (m1_end, m2_end, hue) - green) / i;
		blue_adv = (int) (convert::hls_value (m1_end, m2_end, hue - 120) - blue) / i;

		while (i--)
		{
			*buffer++ = convert::scaled2bgr (red, green, blue);
			red += red_adv;
			green += green_adv;
			blue += blue_adv;
		}
	}

	else
	{
		double	sat, sat_adv;

		sat = 0.0;
		sat_adv = 1.0 / samples;
		while (samples--)
			*buffer++ = convert::bgr (convert::hls2rgb (hue, light, sat)),
			sat += sat_adv;
	}
}

} // blend namespace


#if	defined (_DEBUG)
void
debug_message (
	IN const char *format, ...)
{
	char	buffer [1024];
	va_list	va;
	FILE	*pf;

	// dd/mm/yy + space
	_strdate (buffer);
	buffer [8] = ' ';
	// hh:mi:ss + space
	_strtime (buffer + 9);
	buffer [17] = ' ';
	// thread_id + tab
	sprintf (buffer + 18, "0x%08x", GetCurrentThreadId ());
	buffer [28] = '\t';

	va_start (va, format);
	vsprintf (buffer + 29, format, va);
	strcat (buffer, "\r\n");
	OutputDebugString (buffer);

	pf = fopen ("c:\\xgui.log", "a");
	if (pf)
	{
		fwrite (buffer, strlen (buffer), 1, pf);
		fclose (pf);
	}
}
#endif // _DEBUG


namespace wndclass {

bool reg (
	IN const char *class_name,
	IN WNDCLASS *window_class,
	IN ATOM *class_atom)
{
	*class_atom = RegisterClass (window_class);
	if (*class_atom != 0)
	{
		debug_message ("%s::register_class: done.", class_name);
		return (true);
	}
	debug_message ("%s::register_class: failed.", class_name);
	return (false);
}


bool unreg (
	IN HINSTANCE library_instance,
	IN const char *class_name)
{
	if (UnregisterClass (class_name, library_instance) != 0)
	{
		debug_message ("%s::unregister_class: done.", class_name);
		return (true);
	}
	debug_message ("%s::unregister_class: failed.", class_name);
	return (false);
}

} // wndclass namespace

} // xgui namespace
